﻿using AzObs.Lib.Models;
using AzObs.Lib.Utils;
using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;

namespace AzObs.Lib.Services
{

    /// <summary>
    /// Abstract ObserveService of File Creation.
    /// </summary>
    public abstract class ObserveServiceBase
    {

        private FileSystemWatcher _watcher = null;

        private Control _parentControl;

        private AppConfig _config;

        protected ObserveSetting ObserveSetting { get; private set; }

        /// <summary>
        /// Constractor
        /// </summary>
        /// <param name="jsonFile"></param>
        public ObserveServiceBase(AppConfig config, ObserveSetting observeSetting)
        {
            _config = config;
            ObserveSetting = observeSetting;
        }


        /// <summary>
        /// Switch between Start and Stop Observing of File Creation.
        /// </summary>
        /// <param name="form"></param>
        /// <param name="parentControl"></param>
        /// <returns></returns>
        public bool SwitchObserveRunningState(Form form, Control parentControl) {

            bool isRunning = false;

            _parentControl = parentControl;

            if (_watcher == null)
            { // start observe
                _watcher = new FileSystemWatcher(_config.TargetDir);

                _watcher.NotifyFilter = NotifyFilters.CreationTime
                                        | NotifyFilters.LastWrite
                                        | NotifyFilters.Size;

                _watcher.Changed += new FileSystemEventHandler(watcher_Changed);
                //watcher.Error += OnError;

                _watcher.Filter = _config.TargetExt;
                _watcher.IncludeSubdirectories = false;
                _watcher.EnableRaisingEvents = true;
                _watcher.SynchronizingObject = form;
                isRunning = true;
            }
            else
            { // stop observe 

                _watcher.Dispose();
                _watcher = null;
            }

            return isRunning;
        }


        /// <summary>
        /// Event Handler of detect file creation.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void watcher_Changed(object sender, FileSystemEventArgs e)
        {
            //string value = $"Detected:{e.ChangeType}_{e.FullPath}";
            //Debug.WriteLine(value);
            try
            {
                string readText = File.ReadAllText(e.FullPath, _config.Encoding);
                Reconcile(_parentControl, readText);
            }
            catch (Exception ex) {
                MessageBox.Show(ex.Message, "Error.");
                Environment.Exit(1);
            }
        }


        /// <summary>
        /// Match text to observation data list.
        /// </summary>
        /// <param name="parentControl"></param>
        /// <param name="targetText"></param>
        /// <param name="ovserveItems"></param>
        public virtual void Reconcile(Control parentControl, string targetText)
        {
            foreach (var item in ObserveSetting.ObserveItems)
            {
                if (targetText.Contains(item.MatchWord))
                {
                    var findLabel = (Label)parentControl.Controls[item.Id];

                    if (findLabel != null)
                    {
                        MatchedItemDecorate(findLabel, item);
                    }
                }
            }
        }

        /// <summary>
        /// How to decorate the label of matched item.
        /// </summary>
        /// <param name="findLabel"></param>
        /// <param name="item"></param>
        public virtual void MatchedItemDecorate(Label findLabel, ObserveItem item)
        {
            findLabel.Font = new Font(findLabel.Font, FontStyle.Strikeout);

            if(ObserveSetting.EnableCheckedItemColor) {
                findLabel.ForeColor = Color.Gray;
            }
        }


        /// <summary>
        /// Init display controls.
        /// </summary>
        /// <param name="parentControl"></param>
        public abstract void InitDisplayControls(Control parentControl);


        /// <summary>
        /// Create Label Control by ObserveItem.
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        protected virtual Label CreateItemLabel(ObserveItem item)
        {
            var lbl = new Label();
            lbl.Name = item.Id;
            lbl.Text = item.DispName;
            lbl.ForeColor = ColorUtil.GetColor(item.FontColor);
            lbl.Font = new Font(lbl.Font.OriginalFontName, ObserveSetting.ItemFontSize);

            if (ObserveSetting.ItemFontBold) { 
                lbl.Font = new Font(lbl.Font, FontStyle.Bold);
            }

            lbl.AutoSize = true;

            return lbl;
        }
    }
}
